"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.errorFormatter = exports.removeWarnFormatter = exports.removeInfoFormatter = exports.h5ServeFormatter = exports.formatAtFilename = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const picocolors_1 = __importDefault(require("picocolors"));
const utils_1 = require("../utils");
const constants_1 = require("../constants");
const ast_1 = require("../vite/utils/ast");
const utils_2 = require("../vite/plugins/vitejs/utils");
const SIGNAL_H5_LOCAL = ' > Local:';
const SIGNAL_H5_NETWORK = ' > Network:';
const networkLogs = [];
function formatAtFilename(filename, line, column) {
    return `at ${picocolors_1.default.cyan((0, utils_1.normalizePath)(path_1.default.relative(process.env.UNI_INPUT_DIR, filename.split('?')[0])) +
        ':' +
        (line || 1) +
        ':' +
        (column || 0))}`;
}
exports.formatAtFilename = formatAtFilename;
exports.h5ServeFormatter = {
    test(msg) {
        return msg.includes(SIGNAL_H5_LOCAL) || msg.includes(SIGNAL_H5_NETWORK);
    },
    format(msg) {
        if (msg.includes(SIGNAL_H5_NETWORK)) {
            networkLogs.push(msg);
            process.nextTick(() => {
                if (networkLogs.length) {
                    // 延迟打印所有 network,仅最后一个 network 替换 > 为 -，通知 hbx
                    const len = networkLogs.length - 1;
                    networkLogs[len] = networkLogs[len].replace('>', '-');
                    console.log(networkLogs.join('\n'));
                    networkLogs.length = 0;
                }
            });
            return '';
        }
        return msg.replace('>', '-');
    },
};
const REMOVED_MSGS = [
    'build started...',
    (msg) => {
        return /built in [0-9]+ms\./.test(msg);
    },
    'watching for file changes...',
];
exports.removeInfoFormatter = {
    test(msg) {
        return !!REMOVED_MSGS.find((m) => typeof m === 'string' ? msg.includes(m) : m(msg));
    },
    format() {
        return '';
    },
};
const REMOVED_WARN_MSGS = [];
exports.removeWarnFormatter = {
    test(msg) {
        return !!REMOVED_WARN_MSGS.find((m) => msg.includes(m));
    },
    format() {
        return '';
    },
};
exports.errorFormatter = {
    test(_, opts) {
        return !!(opts && opts.error);
    },
    format(_, opts) {
        return buildErrorMessage(opts.error, [], false);
    },
};
function buildErrorMessage(err, args = [], includeStack = true) {
    var _a, _b;
    if (err.plugin) {
        args.push(`${picocolors_1.default.magenta('[plugin:' + err.plugin + ']')} ${picocolors_1.default.red(err.message)}`);
        if (err.loc &&
            err.hook === 'transform' &&
            err.plugin === 'rollup-plugin-dynamic-import-variables' &&
            err.id &&
            constants_1.EXTNAME_VUE_RE.test(err.id)) {
            try {
                const ast = (0, ast_1.parseVue)(fs_1.default.readFileSync(err.id, 'utf8'), []);
                const scriptNode = ast.children.find((node) => node.type === 1 /* ELEMENT */ && node.tag === 'script');
                if (scriptNode) {
                    const scriptLoc = scriptNode.loc;
                    args.push(picocolors_1.default.yellow(pad((0, utils_2.generateCodeFrame)(scriptLoc.source, err.loc))));
                    // correct error location
                    err.loc.line = scriptLoc.start.line + err.loc.line - 1;
                }
            }
            catch (e) { }
        }
    }
    else {
        args.push(picocolors_1.default.red(err.message));
    }
    if (err.id) {
        args.push(formatAtFilename(err.id, (_a = err.loc) === null || _a === void 0 ? void 0 : _a.line, (_b = err.loc) === null || _b === void 0 ? void 0 : _b.column));
    }
    if (err.frame) {
        args.push(picocolors_1.default.yellow(pad(err.frame)));
    }
    if (includeStack && err.stack) {
        args.push(pad(cleanStack(err.stack)));
    }
    return args.join('\n');
}
function cleanStack(stack) {
    return stack
        .split(/\n/g)
        .filter((l) => /^\s*at/.test(l))
        .join('\n');
}
const splitRE = /\r?\n/;
function pad(source, n = 2) {
    const lines = source.split(splitRE);
    return lines.map((l) => ` `.repeat(n) + l).join(`\n`);
}
