"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.vueI18n = void 0;
const fs_1 = require("fs");
const path_1 = __importDefault(require("path"));
const shared_1 = require("@intlify/shared");
const pluginutils_1 = require("@rollup/pluginutils");
const bundle_utils_1 = require("@intlify/bundle-utils");
const fast_glob_1 = __importDefault(require("fast-glob"));
const debug_1 = __importDefault(require("debug"));
const vite_1 = require("vite");
const query_1 = require("./query");
const debug = debug_1.default('vite-plugin-vue-i18n:index');
const INTLIFY_BUNDLE_IMPORT_ID = '@intlify/vite-plugin-vue-i18n/messages';
const installedPkg = bundle_utils_1.checkInstallPackage('@intlify/vite-plugin-vue-i18n', debug);
function pluginI18n(options = { forceStringify: false }) {
    debug('plugin options:', options);
    // use `normalizePath` for `options.include`
    let include = options.include;
    if (include) {
        if (shared_1.isArray(include)) {
            include = include.map(item => vite_1.normalizePath(item));
        }
        else if (shared_1.isString(include)) {
            include = vite_1.normalizePath(include);
        }
    }
    const filter = pluginutils_1.createFilter(include);
    const runtimeOnly = shared_1.isBoolean(options.runtimeOnly)
        ? options.runtimeOnly
        : true;
    // prettier-ignore
    const compositionOnly = installedPkg === 'vue-i18n'
        ? shared_1.isBoolean(options.compositionOnly)
            ? options.compositionOnly
            : true
        : true;
    // prettier-ignore
    const fullInstall = installedPkg === 'vue-i18n'
        ? shared_1.isBoolean(options.fullInstall)
            ? options.fullInstall
            : true
        : false;
    const defaultSFCLang = shared_1.isString(options.defaultSFCLang)
        ? options.defaultSFCLang
        : undefined;
    const globalSFCScope = !!options.globalSFCScope;
    const useVueI18nImportName = options.useVueI18nImportName;
    if (useVueI18nImportName != null) {
        console.warn(`[vite-plugin-vue-i18n]: 'useVueI18nImportName' option is experimental`);
    }
    const getAliasName = () => installedPkg === 'petite-vue-i18n' &&
        shared_1.isBoolean(useVueI18nImportName) &&
        useVueI18nImportName
        ? 'vue-i18n'
        : `${installedPkg}`;
    const forceStringify = !!options.forceStringify;
    let isProduction = false;
    let sourceMap = false;
    return {
        name: 'vite-plugin-vue-i18n',
        /**
         * NOTE:
         * If we have json (including SFC's custom block),
         * transform it first because it will be transformed into javascript code by `vite:json` plugin.
         */
        enforce: 'pre',
        config(config, { command }) {
            if (command === 'build' && runtimeOnly) {
                normalizeConfigResolveAlias(config);
                if (shared_1.isArray(config.resolve.alias)) {
                    config.resolve.alias.push({
                        find: getAliasName(),
                        replacement: `${installedPkg}/dist/${installedPkg}.runtime.esm-bundler.js`
                    });
                }
                else {
                    // eslint-disable-next-line @typescript-eslint/no-explicit-any
                    ;
                    config.resolve.alias[getAliasName()] = `${installedPkg}/dist/${installedPkg}.runtime.esm-bundler.js`;
                }
                debug(`alias name: ${getAliasName()}`);
                debug(`set ${installedPkg} runtime only: ${installedPkg}/dist/${installedPkg}.runtime.esm-bundler.js`);
            }
            else if (command === 'serve' &&
                installedPkg === 'petite-vue-i18n' &&
                useVueI18nImportName) {
                normalizeConfigResolveAlias(config);
                if (shared_1.isArray(config.resolve.alias)) {
                    config.resolve.alias.push({
                        find: 'vue-i18n',
                        replacement: `petite-vue-i18n/dist/petite-vue-i18n.esm-bundler.js`
                    });
                }
                else {
                    // eslint-disable-next-line @typescript-eslint/no-explicit-any
                    ;
                    config.resolve.alias['vue-i18n'] = `petite-vue-i18n/dist/petite-vue-i18n.esm-bundler.js`;
                }
                debug(`alias name: ${getAliasName()}`);
            }
            config.define = config.define || {};
            config.define['__VUE_I18N_LEGACY_API__'] = !compositionOnly;
            debug(`set __VUE_I18N_LEGACY_API__ is '${config.define['__VUE_I18N_LEGACY_API__']}'`);
            config.define['__VUE_I18N_FULL_INSTALL__'] = fullInstall;
            debug(`set __VUE_I18N_FULL_INSTALL__ is '${config.define['__VUE_I18N_FULL_INSTALL__']}'`);
            config.define['__VUE_I18N_PROD_DEVTOOLS__'] = false;
        },
        configResolved(config) {
            isProduction = config.isProduction;
            sourceMap = config.command === 'build' ? !!config.build.sourcemap : false;
            // json transform handling
            const jsonPlugin = config.plugins.find(p => p.name === 'vite:json');
            if (jsonPlugin) {
                const orgTransform = jsonPlugin.transform; // backup @rollup/plugin-json
                jsonPlugin.transform = async function (code, id) {
                    if (!/\.json$/.test(id) || filter(id)) {
                        return;
                    }
                    /**
                     * NOTE:
                     * `vite:json` plugin will be handled if the query generated from the result of parse SFC
                     * with `vite:vue` plugin contains json as follows.
                     * e.g src/components/HelloI18n.vue?vue&type=i18n&index=1&lang.json
                     *
                     * To avoid this, return the result that has already been processed (`enforce: 'pre'`) in the wrapped json plugin.
                     */
                    const { query } = query_1.parseVueRequest(id);
                    if (query.vue) {
                        return;
                    }
                    debug('org json plugin');
                    return orgTransform.apply(this, [code, id]);
                };
            }
        },
        resolveId(id) {
            if (id === INTLIFY_BUNDLE_IMPORT_ID) {
                return id;
            }
        },
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        async load(id, ssr) {
            if (id === INTLIFY_BUNDLE_IMPORT_ID && include) {
                let resourcePaths = [];
                const includePaths = shared_1.isArray(include) ? include : [include];
                for (const inc of includePaths) {
                    resourcePaths = [...resourcePaths, ...(await fast_glob_1.default(inc))];
                }
                resourcePaths = resourcePaths.filter((el, pos) => resourcePaths.indexOf(el) === pos);
                const code = await generateBundleResources(resourcePaths, isProduction, forceStringify);
                return {
                    code,
                    map: { mappings: '' }
                };
            }
        },
        async handleHotUpdate({ file, server }) {
            if (/\.(json5?|ya?ml)$/.test(file)) {
                const module = server.moduleGraph.getModuleById(INTLIFY_BUNDLE_IMPORT_ID);
                if (module) {
                    server.moduleGraph.invalidateModule(module);
                    return [module];
                }
            }
        },
        async transform(code, id) {
            const { filename, query } = query_1.parseVueRequest(id);
            debug('transform', id, JSON.stringify(query));
            let langInfo = 'json';
            let inSourceMap;
            if (!query.vue) {
                if (/\.(json5?|ya?ml)$/.test(id) && filter(id)) {
                    langInfo = path_1.default.parse(filename).ext;
                    const generate = /\.?json5?/.test(langInfo)
                        ? bundle_utils_1.generateJSON
                        : bundle_utils_1.generateYAML;
                    const parseOptions = getOptions(filename, isProduction, query, sourceMap, inSourceMap, globalSFCScope, forceStringify);
                    debug('parseOptions', parseOptions);
                    const { code: generatedCode, map } = generate(code, parseOptions);
                    debug('generated code', generatedCode, id);
                    debug('sourcemap', map, id);
                    if (code === generatedCode)
                        return;
                    return {
                        code: generatedCode,
                        map: (sourceMap ? map : { mappings: '' }) // eslint-disable-line @typescript-eslint/no-explicit-any
                    };
                }
            }
            else {
                // for Vue SFC
                if (isCustomBlock(query)) {
                    if (shared_1.isString(query.lang)) {
                        langInfo = query.src
                            ? query.lang === 'i18n'
                                ? 'json'
                                : query.lang
                            : query.lang;
                    }
                    else if (defaultSFCLang) {
                        langInfo = defaultSFCLang;
                    }
                    const generate = /\.?json5?/.test(langInfo)
                        ? bundle_utils_1.generateJSON
                        : bundle_utils_1.generateYAML;
                    const parseOptions = getOptions(filename, isProduction, query, sourceMap, inSourceMap, globalSFCScope, forceStringify);
                    debug('parseOptions', parseOptions);
                    const { code: generatedCode, map } = generate(code, parseOptions);
                    debug('generated code', generatedCode, id);
                    debug('sourcemap', map, id);
                    if (code === generatedCode)
                        return;
                    return {
                        code: generatedCode,
                        map: (sourceMap ? map : { mappings: '' }) // eslint-disable-line @typescript-eslint/no-explicit-any
                    };
                }
            }
        }
    };
}
function normalizeConfigResolveAlias(config) {
    // NOTE: cannot resolve Optional Chaining in jest E2E ...
    if (config.resolve && config.resolve.alias) {
        return;
    }
    if (!config.resolve) {
        config.resolve = { alias: [] };
    }
    else if (!config.resolve.alias) {
        config.resolve.alias = [];
    }
}
async function getRaw(path) {
    return fs_1.promises.readFile(path, { encoding: 'utf-8' });
}
function isCustomBlock(query) {
    // NOTE: should be more improvement. difference query type and blocktype in some environment ...
    return (!shared_1.isEmptyObject(query) &&
        'vue' in query &&
        (query['type'] === 'custom' ||
            query['type'] === 'i18n' ||
            query['blockType'] === 'i18n'));
}
function getOptions(filename, isProduction, query, sourceMap, inSourceMap, isGlobal = false, forceStringify = false) {
    const mode = isProduction ? 'production' : 'development';
    const baseOptions = {
        filename,
        sourceMap,
        inSourceMap,
        forceStringify,
        env: mode,
        onWarn: (msg) => {
            console.warn(`[vite-plugin-vue-i18n]: ${filename} ${msg}`);
        },
        onError: (msg) => {
            console.error(`[vite-plugin-vue-i18n]: ${filename} ${msg}`);
        }
    };
    if (isCustomBlock(query)) {
        return Object.assign(baseOptions, {
            type: 'sfc',
            locale: shared_1.isString(query.locale) ? query.locale : '',
            isGlobal: isGlobal || query.global != null
        });
    }
    else {
        return Object.assign(baseOptions, {
            type: 'plain',
            isGlobal: false
        });
    }
}
async function generateBundleResources(resources, isProduction, forceStringify, isGlobal = false) {
    const codes = [];
    for (const res of resources) {
        debug(`${res} bundle loading ...`);
        if (/\.(json5?|ya?ml)$/.test(res)) {
            const { ext, name } = path_1.default.parse(res);
            const source = await getRaw(res);
            const generate = /json5?/.test(ext) ? bundle_utils_1.generateJSON : bundle_utils_1.generateYAML;
            const parseOptions = getOptions(res, isProduction, {}, false, undefined, isGlobal, forceStringify);
            parseOptions.type = 'bare';
            const { code } = generate(source, parseOptions);
            debug('generated code', code);
            codes.push(`${JSON.stringify(name)}: ${code}`);
        }
    }
    return `export default {
  ${codes.join(`,\n`)}
}`;
}
// overwrite for cjs require('...')() usage
exports.default = pluginI18n;
exports.vueI18n = pluginI18n;
