"use strict";
/**
 * Code generator for i18n json/json5 resource
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.generate = void 0;
const jsonc_eslint_parser_1 = require("jsonc-eslint-parser");
const shared_1 = require("@intlify/shared");
const codegen_1 = require("./codegen");
/**
 * @internal
 */
function generate(targetSource, { type = 'plain', bridge = false, filename = 'vue-i18n-loader.json', inSourceMap = undefined, locale = '', isGlobal = false, sourceMap = false, env = 'development', forceStringify = false, useClassComponent = false }, injector) {
    const target = Buffer.isBuffer(targetSource)
        ? targetSource.toString()
        : targetSource;
    // const value = JSON.stringify(JSON.parse(target))
    //   .replace(/\u2028/g, '\\u2028') // line separator
    //   .replace(/\u2029/g, '\\u2029') // paragraph separator
    const value = target;
    const options = {
        type,
        bridge,
        source: value,
        sourceMap,
        locale,
        isGlobal,
        inSourceMap,
        env,
        filename,
        forceStringify,
        useClassComponent
    };
    const generator = codegen_1.createCodeGenerator(options);
    const ast = jsonc_eslint_parser_1.parseJSON(value, { filePath: filename });
    const codeMaps = generateNode(generator, ast, options, injector);
    const { code, map } = generator.context();
    // if (map) {
    //   const s = new SourceMapConsumer((map as any).toJSON())
    //   s.eachMapping(m => {
    //     console.log('sourcemap json', m)
    //   })
    // }
    // prettier-ignore
    const newMap = map
        ? codegen_1.mapLinesColumns(map.toJSON(), codeMaps, inSourceMap) || null // eslint-disable-line @typescript-eslint/no-explicit-any
        : null;
    return {
        ast,
        code,
        map: newMap != null ? newMap : undefined
    };
}
exports.generate = generate;
function generateNode(generator, node, options, injector) {
    const propsCountStack = [];
    const itemsCountStack = [];
    const { forceStringify } = generator.context();
    const codeMaps = new Map();
    const { type, bridge, sourceMap, isGlobal, locale, useClassComponent } = options;
    const componentNamespace = bridge
        ? `Component.options`
        : useClassComponent
            ? `Component.__o`
            : `Component`;
    jsonc_eslint_parser_1.traverseNodes(node, {
        enterNode(node, parent) {
            switch (node.type) {
                case 'Program':
                    if (type === 'plain') {
                        generator.push(`export default `);
                    }
                    else if (type === 'sfc') {
                        // for 'sfc'
                        const variableName = type === 'sfc' ? (!isGlobal ? '__i18n' : '__i18nGlobal') : '';
                        const localeName = type === 'sfc' ? (locale != null ? locale : `""`) : '';
                        const exportSyntax = bridge ? `module.exports =` : `export default`;
                        generator.push(`${exportSyntax} function (Component) {`);
                        generator.indent();
                        generator.pushline(`${componentNamespace}.${variableName} = ${componentNamespace}.${variableName} || []`);
                        generator.push(`${componentNamespace}.${variableName}.push({`);
                        generator.indent();
                        generator.pushline(`"locale": ${JSON.stringify(localeName)},`);
                        generator.push(`"resource": `);
                    }
                    break;
                case 'JSONObjectExpression':
                    generator.push(`{`);
                    generator.indent();
                    propsCountStack.push(node.properties.length);
                    if (parent.type === 'JSONArrayExpression') {
                        const lastIndex = itemsCountStack.length - 1;
                        itemsCountStack[lastIndex] = --itemsCountStack[lastIndex];
                    }
                    break;
                case 'JSONProperty':
                    if (node.value.type === 'JSONLiteral' &&
                        (node.key.type === 'JSONLiteral' ||
                            node.key.type === 'JSONIdentifier')) {
                        const name = node.key.type === 'JSONLiteral' ? node.key.value : node.key.name;
                        const value = node.value.value;
                        if (shared_1.isString(value)) {
                            generator.push(`${JSON.stringify(name)}: `);
                            const { code, map } = codegen_1.generateMessageFunction(value, options);
                            sourceMap && map != null && codeMaps.set(value, map);
                            generator.push(`${code}`, node.value, value);
                        }
                        else {
                            if (forceStringify) {
                                const strValue = JSON.stringify(value);
                                generator.push(`${JSON.stringify(name)}: `);
                                const { code, map } = codegen_1.generateMessageFunction(strValue, options);
                                sourceMap && map != null && codeMaps.set(strValue, map);
                                generator.push(`${code}`, node.value, strValue);
                            }
                            else {
                                generator.push(`${JSON.stringify(name)}: ${JSON.stringify(value)}`);
                            }
                        }
                    }
                    else if ((node.value.type === 'JSONObjectExpression' ||
                        node.value.type === 'JSONArrayExpression') &&
                        (node.key.type === 'JSONLiteral' ||
                            node.key.type === 'JSONIdentifier')) {
                        const name = node.key.type === 'JSONLiteral' ? node.key.value : node.key.name;
                        generator.push(`${JSON.stringify(name)}: `);
                    }
                    const lastIndex = propsCountStack.length - 1;
                    propsCountStack[lastIndex] = --propsCountStack[lastIndex];
                    break;
                case 'JSONArrayExpression':
                    generator.push(`[`);
                    generator.indent();
                    itemsCountStack.push(node.elements.length);
                    break;
                case 'JSONLiteral':
                    if (parent.type === 'JSONArrayExpression') {
                        if (node.type === 'JSONLiteral') {
                            const value = node.value;
                            if (shared_1.isString(value)) {
                                const { code, map } = codegen_1.generateMessageFunction(value, options);
                                sourceMap && map != null && codeMaps.set(value, map);
                                generator.push(`${code}`, node, value);
                            }
                            else {
                                if (forceStringify) {
                                    const strValue = JSON.stringify(value);
                                    const { code, map } = codegen_1.generateMessageFunction(strValue, options);
                                    sourceMap && map != null && codeMaps.set(strValue, map);
                                    generator.push(`${code}`, node, strValue);
                                }
                                else {
                                    generator.push(`${JSON.stringify(value)}`);
                                }
                            }
                        }
                        const lastIndex = itemsCountStack.length - 1;
                        itemsCountStack[lastIndex] = --itemsCountStack[lastIndex];
                    }
                    break;
                default:
                    break;
            }
        },
        leaveNode(node, parent) {
            switch (node.type) {
                case 'Program':
                    if (type === 'sfc') {
                        generator.deindent();
                        generator.push(`})`);
                        if (bridge && injector) {
                            generator.newline();
                            generator.pushline(`${componentNamespace}.__i18nBridge = ${componentNamespace}.__i18nBridge || []`);
                            generator.pushline(`${componentNamespace}.__i18nBridge.push('${injector()}')`);
                            generator.pushline(`delete ${componentNamespace}._Ctor`);
                        }
                        generator.deindent();
                        generator.pushline(`}`);
                    }
                    break;
                case 'JSONObjectExpression':
                    if (propsCountStack[propsCountStack.length - 1] === 0) {
                        propsCountStack.pop();
                    }
                    generator.deindent();
                    generator.push(`}`);
                    if (parent.type === 'JSONArrayExpression') {
                        if (itemsCountStack[itemsCountStack.length - 1] !== 0) {
                            generator.pushline(`,`);
                        }
                    }
                    break;
                case 'JSONProperty':
                    if (propsCountStack[propsCountStack.length - 1] !== 0) {
                        generator.pushline(`,`);
                    }
                    break;
                case 'JSONArrayExpression':
                    if (itemsCountStack[itemsCountStack.length - 1] === 0) {
                        itemsCountStack.pop();
                    }
                    generator.deindent();
                    generator.push(`]`);
                    if (parent.type === 'JSONArrayExpression') {
                        if (itemsCountStack[itemsCountStack.length - 1] !== 0) {
                            generator.pushline(`,`);
                        }
                    }
                    break;
                case 'JSONLiteral':
                    if (parent.type === 'JSONArrayExpression') {
                        if (itemsCountStack[itemsCountStack.length - 1] !== 0) {
                            generator.pushline(`,`);
                        }
                        else {
                            generator.pushline(`,`);
                        }
                    }
                    break;
                default:
                    break;
            }
        }
    });
    return codeMaps;
}
